using System.Windows.Forms;
using Microsoft.VisualBasic;
using System;
using MicroFour.StrataFrame.Business;
using StrataFlix.Business;
using System.Collections.Generic;
using System.Data.SqlClient;
using System.Drawing;


namespace StrataFlix
{
	public partial class PeopleMaintenance
	{
		
		#region " Constructors "
		
		/// <summary>
		/// Default constructor
		/// </summary>
		/// <remarks></remarks>
		public PeopleMaintenance()
		{
			
			// This call is required by the Windows Form Designer.
			InitializeComponent();
			
			//-- Load the images
			LoadImages();
			
			//-- Ensure that the home page is initially shown
			pmMain.SetActivePage(PageHome);
			
			
		}
		
		#endregion
		
		#region " Handled Events "
		
		
		private void action_Print_Click()
		{
			Reporting.ReportEngine.RunReport(Reporting.StrataFlixReports.PeopleProfile);
		}
		
		private void action_Delete_Click()
		{
			DeleteRecord();
		}
		
		private void action_Add_Click()
		{
			AddRecord();
		}
		
		private void lnkAddMovie_LinkClicked(object sender, System.Windows.Forms.LinkLabelLinkClickedEventArgs e)
		{
			AddRecord();
		}
		
		private void action_ClearPosterImage_Click()
		{
			ClearPhoto();
		}
		
		private void action_AttachPosterImage_Click()
		{
			AttachPhoto();
		}
		
		private void MovieMaintenance_AfterUndo(MicroFour.StrataFrame.Data.AfterSaveUndoByFormEventArgs e)
		{
			//-- See if there are any records within the people BO
			if (People.Count > 0)
			{
				//-- Reload the data from the server
				LoadAllPeopleData(People.pl_pk, true);
				
				//-- Requery the lists
				lstMovies.Requery();
				
				//-- Set the object states
				SetObjectStates();
			}
			else
			{
				//-- Force the display back to the getting started page
				pmMain.SetActivePage(PageHome);
			}
		}
		
		private void lstMovies_ListPopulating(MicroFour.StrataFrame.UI.ListPopulatingEventArgs e)
		{
			e.Parameters[0].Value = MovieCast;
			e.Parameters[1].Value = MicroFour.StrataFrame.Business.BusinessCloneDataType.ClearAndFillFromDefaultView;
		}
		
		private void lstMovies_RowPopulating(MicroFour.StrataFrame.UI.Windows.Forms.RowPopulatingEventArgs e)
		{
			//-- In this example, all of the images will be the same, so we will just force it to the first item in the list
			e.ImageIndex = 0;
		}
		
		private void PageHome_PageActivated(MicroFour.StrataFrame.UI.Windows.Forms.PanelPageEventArgs e)
		{
			SetObjectStates();
		}
		
		private void action_GettingStarted_Click()
		{
			pmMain.SetActivePage(PageHome);
		}
		
		private void PageEditor_PageActivated(MicroFour.StrataFrame.UI.Windows.Forms.PanelPageEventArgs e)
		{
			SetObjectStates();
		}
		
		private void action_Search_Click()
		{
			SearchForRecord();
		}
		
		private void lnkFindMovie_LinkClicked(System.Object sender, System.Windows.Forms.LinkLabelLinkClickedEventArgs e)
		{
			SearchForRecord();
		}
		
		#endregion
		
		#region " Private Methods "
		
		/// <summary>
		/// Adds a new record
		/// </summary>
		/// <remarks></remarks>
		private void AddRecord()
		{
			//-- Add the new movie record
			People.Add();
			
			//-- Clear any BOs that may have been filled from previous records
			MovieCast.Clear();
			
			//-- Ensure that the editor page is active
			if (!(pmMain.CurrentPage.Equals(PageEditor)))
			{
				pmMain.SetActivePage(PageEditor);
			}
			
			//-- Requery the lists
			lstMovies.Requery();
			
			//-- Set the object states
			SetObjectStates();
			
			//-- Refresh the image
			//imgMoviePoster.Image = MovieImages.mi_PosterLarge
			
			//-- Set the focus to the title field
			txtCode.Focus();
		}
		
		/// <summary>
		/// Attaches a movie poster image
		/// </summary>
		/// <remarks></remarks>
		private void AttachPhoto()
		{
			if (this.selectPhoto.ShowDialog() == System.Windows.Forms.DialogResult.OK)
			{
				//-- Ensure that a image record exists
				if (PeopleImages.Count == 0)
				{
					PeopleImages.NewRow();
				}
				
				//-- Set the Photo
				PeopleImages.pli_PhotoLarge = (Image) (new Bitmap(selectPhoto.FileName));
				
				//-- Update the object states
				SetObjectStates();
				
				//-- Refresh the binding
				PeopleImages.Refresh();
			}
		}
		
		/// <summary>
		/// Deletes the currently loaded record
		/// </summary>
		/// <remarks></remarks>
		private void DeleteRecord()
		{
			if (MicroFour.StrataFrame.Messaging.MessageForm.ShowMessageByKey("PeopleMaintenance_Delete", People.pl_FullName) == MicroFour.StrataFrame.Messaging.MessageFunctionType.Yes)
			{
				//-- Delete the record
				People.DeleteCurrentRow(false);
				
				//-- Return to the getting started page
				this.pmMain.SetActivePage(PageHome);
			}
		}
		
		/// <summary>
		/// Clears the attached photo
		/// </summary>
		/// <remarks></remarks>
		private void ClearPhoto()
		{
			//-- Clear out the image
			PeopleImages.pli_ImageData = new byte[] {};
			
			//-- Reset the object states
			SetObjectStates();
			
			//-- Refresh the binding for the image
			PeopleImages.Refresh();
		}
		
		/// <summary>
		/// Retrieves all of the data that will be used within this dialog
		/// </summary>
		/// <remarks></remarks>
		private void LoadAllPeopleData(int peoplePk, bool includePeopleRecord)
		{
			//-- Perform the query and retrieve the data
			if (includePeopleRecord)
			{
				BusinessLayer.FillMultipleDataTables(ApplicationQueries.PeopleMaintenance_AllData(peoplePk, includePeopleRecord), People, MovieCast, PeopleImages);
			}
			else
			{
				BusinessLayer.FillMultipleDataTables(ApplicationQueries.PeopleMaintenance_AllData(peoplePk, includePeopleRecord), MovieCast, PeopleImages);
			}
			
			//-- Reset the image to the default.  The reason this is being done in code instead of through binding
			//   is because when a BO doesn't have any rows, the bindings are removed. Yet we still want to show that there is no
			//   photo (or the default photo).
			if (PeopleImages.Count == 0)
			{
				imgPhoto.Image = PeopleImages.pli_PhotoLarge;
			}
		}
		
		/// <summary>
		/// Loads the images from the resource files into the image list. The reason this is done
		/// in code is to improve the quality of the images over multiple compiles.  When added to the ImageList
		/// control through the designer the control will reduce the quality of the images after a number of compiles
		/// which will then begin to make the images look grainly and as though the edges bleed into the background instead
		/// of remaining crisp.  One way to prevent this from happening is to manually load the images into the image list
		/// at run-time which will prevent this from happening.
		/// </summary>
		/// <remarks></remarks>
		private void LoadImages()
		{
			imgImages.Images.Add("Movie", global::StrataFlix.Properties.PeopleResources.movie_16);
		}
		
		/// <summary>
		/// Performs a search for a particular record
		/// </summary>
		/// <remarks></remarks>
		private void SearchForRecord()
		{
			//-- Show the browse dialog
			if (PeopleBrowseDialog1.ShowDialog() == System.Windows.Forms.DialogResult.OK)
			{
				//-- Retrieve all of the people data
				LoadAllPeopleData(People.pl_pk, false);
				
				//-- Reload the movie list
				lstMovies.Requery();
				
				//-- Since a record was selected, show the editor page
				pmMain.SetActivePage(PageEditor);
			}
		}
		
		#endregion
		
		#region " Protected Methods "
		
		/// <summary>
		/// Sets the objects state of items within this dialog
		/// </summary>
		/// <remarks></remarks>
		protected override void SetObjectStates()
		{
			//-- Establish Locals
			bool isDirty = People.IsDirty || MovieCast.IsDirty || PeopleImages.IsDirty;
			
			//-- Set the actions
			action_GettingStarted.Enabled = (! isDirty) && (!(pmMain.CurrentPage.Equals(PageHome)));
			action_Add.Enabled = ! isDirty;
			action_Search.Enabled = ! isDirty;
			action_Print.Enabled = ! isDirty;
			action_Delete.Enabled = ! isDirty && (pmMain.CurrentPage.Equals(PageEditor));
			action_AttachPhoto.Visible = People.Count > 0 && (pmMain.CurrentPage.Equals(PageEditor));
			action_ClearPhotoImage.Visible = People.Count > 0 && (pmMain.CurrentPage.Equals(PageEditor)) && PeopleImages.HasPhoto();
			
			
			//-- Determine if the additional separator should be displayed
			sep_Additional.Visible = action_AttachPhoto.Visible || action_ClearPhotoImage.Visible;
		}
		
		#endregion
		
	}
	
}
